// Services/FindDriverService.js

const User = require("../Models/UserModel");
const Order = require("../Models/OrderModel");
const orderEvents = require("../Events/orderEvents");
const { sendFCM } = require("../Helpers/fcmHelper");
const { sendToUuid } = require("../Helpers/WsSender");

const runningLoops = new Map();

async function startFindDriverLoop(orderID, data) {
  console.log(`▶️ startFindDriverLoop(${orderID})`);
  console.log("order:", data);
  if (!orderID || !data) return false;

  if (runningLoops.has(orderID)) {
    console.log(`♻️ Restart loop for ${orderID}`);
    runningLoops.delete(orderID);
  }

  const store = {
    accepted: false,
    rejectedDrivers: new Set(),
    startTime: Date.now(),
  };

  runningLoops.set(orderID, store);

  async function processLoop() {
    if (store.accepted) return;

    // Timeout
    if (Date.now() - store.startTime > 30000) {
      console.log(`⏱ Timeout order ${orderID}`);
      await Order.cancelOrder(orderID);
      runningLoops.delete(orderID);
      return;
    }

    const nearby = await Order.nearbyDriver(
      data.customer_phone,
      data.action,
      5000,
      data.payment_metode
    );

    if (!nearby || nearby.length === 0) {
      console.log("❌ Tidak ada driver, retry 5 detik...");
      return setTimeout(processLoop, 5000);
    }

    if (store.rejectedDrivers.size >= nearby.length) {
      console.log("♻️ Reset reject list");
      store.rejectedDrivers.clear();
    }

    for (const driver of nearby) {
      if (store.accepted) return;
      if (store.rejectedDrivers.has(driver.phone)) continue;

      await Order.accepttOrder(driver.phone, driver.nama, orderID, "new");

      sendToUuid(driver.uuid, {
        type: "server_message",
        action:
          data.order_type === "food" ? "food_order_new" : "ride_order_new",
        message: data,
      });

      await sendFCM(driver.fcm_token, {
        title: "Pesanan Baru",
        body: `Order #${orderID}`,
      });

      await new Promise((resolve) => {
        let responded = false;

        orderEvents.once(`reject_${orderID}_${driver.phone}`, () => {
          store.rejectedDrivers.add(driver.phone);
          responded = true;
          resolve();
        });

        orderEvents.once(`accept_${orderID}_${driver.phone}`, () => {
          store.accepted = true;
          responded = true;
          resolve();
        });

        setTimeout(() => {
          if (!responded) {
            store.rejectedDrivers.add(driver.phone);
            resolve();
          }
        }, 15000);
      });
    }

    if (!store.accepted) {
      return setTimeout(processLoop, 3000);
    }
  }

  processLoop();
  return true;
}

// Wrapper agar bisa dipanggil dari API
function invokeFindDriverLoop(params) {
  if (!params.order_id || !params.data) return false;
  return startFindDriverLoop(params.order_id, params.data);
}

// EXPORT WAJIB BEGINI
module.exports = {
  startFindDriverLoop,
  invokeFindDriverLoop,
};
