const db = require("../Config/Database");
const User = require("../Models/UserModel");
const mysql = require("mysql2");

module.exports = {
  /**
   * Ambil daftar driver dalam radius 500 m dari lokasi user (berdasarkan phone)
   * @param {string} phone
   */
  async nearbyDriver0(phone, order_type, radius, payment_metode) {
    if (!order_type) throw new Error("order_type tidak boleh kosong");
    if (!radius) radius = 5000;

    const [userRows] = await User.getByPhone(phone);
    if (userRows.length === 0) {
      throw new Error("User dengan phone " + phone + " tidak ditemukan");
    }

    const userLat = userRows[0].loc_lat;
    const userLng = userRows[0].loc_lng;

    // 🔹 Buat kondisi saldo dinamis tergantung metode pembayaran
    const saldoCondition =
      payment_metode?.toLowerCase() === "tunai" ? "AND A.saldo >= 0" : ""; // jika saldo, hapus filter saldo

    let sql = "";

    sql = `
      SELECT 
          A.id,
          A.uuid,
          A.nama,
          A.phone,
          A.loc_lat,
          A.loc_lng,
          A.fcm_token,
          ST_Distance_Sphere(
              POINT(A.loc_lng, A.loc_lat),
              POINT(?, ?)
          ) AS distance
      FROM users A
      WHERE ST_Distance_Sphere(
                POINT(A.loc_lng, A.loc_lat),
                POINT(?, ?)
            ) <= ${radius}
        AND A.phone <> ?
        AND A.status_aktif = 'Y'
        AND A.login = 'Y'
        AND A.login_wss = 'Y'
        AND A.driver_ride = 'Y'
        AND A.order_status = '0'
        AND A.role = 'driver'      
        ${saldoCondition}
      ORDER BY distance ASC;
    `;

    const [rows] = await db.query(sql, [
      userLng,
      userLat,
      userLng,
      userLat,
      phone,
    ]);

    return rows; // array driver terdekat
  },

  async nearbyDriver1(phone, order_type, radius, payment_metode) {
    if (!order_type) throw new Error("order_type tidak boleh kosong");
    if (!radius) radius = 5000;

    const [userRows] = await User.getByPhone(phone);
    if (userRows.length === 0) {
      throw new Error("User dengan phone " + phone + " tidak ditemukan");
    }

    const userLat = userRows[0].loc_lat;
    const userLng = userRows[0].loc_lng;

    // 🔹 Tentukan driver field berdasarkan order_type
    let driverField = "A.driver_ride"; // default
    if (order_type === "food") {
      driverField = "A.driver_food";
    }

    // 🔹 Kondisi saldo (tunai vs saldo)
    const saldoCondition =
      payment_metode?.toLowerCase() === "tunai" ? "AND A.saldo >= 0" : "";

    const sql = `
    SELECT 
        A.id,
        A.uuid,
        A.nama,
        A.phone,
        A.loc_lat,
        A.loc_lng,
        A.fcm_token,
        ST_Distance_Sphere(
            POINT(A.loc_lng, A.loc_lat),
            POINT(?, ?)
        ) AS distance
    FROM users A
    WHERE ST_Distance_Sphere(
              POINT(A.loc_lng, A.loc_lat),
              POINT(?, ?)
          ) <= ${radius}
      AND A.phone <> ?
      AND A.status_aktif = 'Y'
      AND A.login = 'Y'
      AND A.login_wss = 'Y'
      AND ${driverField} = 'Y'
      AND A.order_status = '0'
      AND A.role = 'driver'
      ${saldoCondition}
    ORDER BY distance ASC
  `;

    console.log(`radius: ${radius}`);
    console.log(`driverField: ${driverField}`);
    console.log(`saldoCondition: ${saldoCondition}`);
    const [rows] = await db.query(sql, [
      userLng,
      userLat,
      userLng,
      userLat,
      phone,
    ]);

    return rows;
  },
  async nearbyDriver(
    phone,
    order_type,
    radius,
    payment_metode,
    RestaurantLocLat,
    RestaurantLocLng
  ) {
    if (!order_type) throw new Error("order_type tidak boleh kosong");
    if (!radius) radius = 5000;

    const [userRows] = await User.getByPhone(phone);
    if (userRows.length === 0) {
      throw new Error("User dengan phone " + phone + " tidak ditemukan");
    }

    const userLat = userRows[0].loc_lat;
    const userLng = userRows[0].loc_lng;

    let driverField = "A.driver_ride";

    if (order_type === "food") {
      driverField = "A.driver_food";
    }

    const saldoCondition =
      payment_metode?.toLowerCase() === "tunai" ? "AND A.saldo >= 0" : "";

    const sql = `
    SELECT 
        A.id,
        A.uuid,
        A.nama,
        A.phone,
        A.loc_lat,
        A.loc_lng,
        A.fcm_token,
        ST_Distance_Sphere(
            POINT(A.loc_lng, A.loc_lat),
            POINT(?, ?)
        ) AS distance
    FROM users A
    WHERE ST_Distance_Sphere(
              POINT(A.loc_lng, A.loc_lat),
              POINT(?, ?)
          ) <= ${radius}
    
      AND A.status_aktif = 'Y'
      AND A.login = 'Y'
      AND A.login_wss = 'Y'
      AND ${driverField} = '1'
      AND A.order_status = '0'
      AND A.role = 'driver'
      ${saldoCondition}
    ORDER BY distance ASC
  `;

    let params;
    if (order_type === "food") {
      params = [
        RestaurantLocLng,
        RestaurantLocLat,
        RestaurantLocLng,
        RestaurantLocLat,
      ];
    } else if (order_type === "ride") {
      params = [userLng, userLat, userLng, userLat, phone];
    } else {
      return;
    }

    // 🔥 LOG SQL FINAL
    const mysql = require("mysql2");
    const finalSql = mysql.format(sql, params);

    console.log("========== SQL FINAL ==========");
    console.log(finalSql);
    console.log("================================");

    const [rows] = await db.query(sql, params);
    return rows;
  },

  async insertOrder(
    order_id,
    order_type,
    customer_phone,
    customer_name,
    driver_phone,
    pickup,
    destination,
    pickup_lat,
    pickup_lng,
    destination_lat,
    destination_lng,
    distance,
    duration,
    tarif,
    keterangan,
    payment_metode
  ) {
    await db.query(
      "INSERT into orders (order_id,order_type,customer_phone,customer_name,driver_phone,pickup,destination,pickup_lat,pickup_lng,destination_lat,destination_lng,distance,duration,tarif,keterangan,payment_metode) values (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)",
      [
        order_id,
        order_type,
        customer_phone,
        customer_name,
        driver_phone,
        pickup,
        destination,
        pickup_lat,
        pickup_lng,
        destination_lat,
        destination_lng,
        distance,
        duration,
        tarif,
        keterangan,
        payment_metode,
      ]
    );

    // await db.query('UPDATE users set order_status=? WHERE phone=?', ["1", driver_phone]);
  },

  async generateOrderId() {
    const sql = `
    SELECT CONCAT(
      'R-',
      LPAD(
        IFNULL(MAX(CAST(SUBSTRING(order_id, 3) AS UNSIGNED)), 0) + 1,
        7,
        '0'
      )
    ) AS next_order_id
    FROM orders
  `;

    const [rows] = await db.query(sql);

    return rows[0]?.next_order_id || "R-0000001";
  },

  async rejectOrder(phone, order_id) {
    // await db.query('UPDATE orders set order_status=? WHERE order_id=?', ["9", order_id]);
    await db.query("UPDATE users set order_status=? WHERE phone=?", [
      "0",
      phone,
    ]);
  },

  async accepttOrder(driverPhone, driverName, order_id, order_status, tarif) {
    // console.log(`order_status : ${order_status} ${order_id} ${driverPhone}`);

    if (driverPhone == "") {
      await db.query(
        "UPDATE users set order_status=?,order_id=? WHERE order_id=?",
        [order_status, order_id, order_id]
      );
    } else {
      if (order_status == "searching") {
        await db.query(
          "UPDATE users set order_status=?,order_id=? WHERE phone=?",
          ["new", order_id, driverPhone]
        );
      } else {
        await db.query(
          "UPDATE users set order_status=?,order_id=? WHERE phone=?",
          [order_status, order_id, driverPhone]
        );
      }
    }

    await db.query(
      "UPDATE orders set driver_phone=?, driver_name=?,  order_status=? WHERE order_id=?",
      [driverPhone, driverName, order_status, order_id]
    );

    if (order_status == "completed") {
      console.log(`TARIf: ${tarif}`);
      await db.query(
        "UPDATE users set order_status=?,order_id=? WHERE order_id=?",
        ["0", "", order_id]
      );
    }
  },
  async getByOrderID0(orderID) {
    return db.query("SELECT * FROM orders WHERE order_id = ?", [orderID]);
  },
  async getByOrderID(orderID) {
    return db.query(
      `
    SELECT 
      o.*,

      -- 📍 Data Customer
      c.nama AS customer_name,
      c.phone AS customer_phone,
      c.uuid AS uuid_customer,
      c.fcm_token AS fcm_token, 

      -- 📍 Data Driver
      d.nama AS driver_name,
      d.phone AS driver_phone,
      d.driver_tipe_kendaraan,
      d.driver_no_pol,
      d.uuid AS uuid_driver

    FROM orders o
    LEFT JOIN users d ON d.order_id = o.order_id
    LEFT JOIN users c ON c.phone = o.customer_phone
    WHERE o.order_id = ?
  `,
      [orderID]
    );
  },
  async getByOrderIDResto(orderID) {
    return db.query(
      `
    SELECT 
    o.*,

    -- 📍 Customer
    c.nama AS customer_name,
    c.phone AS customer_phone,
    c.uuid AS uuid_customer,
    c.fcm_token AS customer_fcm_token,

    -- 📍 Driver
    d.nama AS driver_name,
    d.phone AS driver_phone,
    d.driver_tipe_kendaraan,
    d.driver_no_pol,
    d.uuid AS uuid_driver,

    -- 📍 Restoran
    r.name AS restaurant_name,
    r.loc_lat AS restaurant_loc_lat,
    r.loc_lng AS restaurant_loc_lng,
    ru.phone AS restaurant_phone,
    ru.uuid AS restaurant_uuid,
    ru.fcm_token AS restaurant_fcm_token

FROM orders o

-- JOIN DRIVER
LEFT JOIN users d 
    ON d.order_id = o.order_id 
   AND d.role = 'driver'

-- JOIN CUSTOMER
LEFT JOIN users c 
    ON c.phone = o.customer_phone

-- JOIN TABEL RESTAURANT
LEFT JOIN restaurants r 
    ON r.id = o.restaurant_id

-- JOIN USERS (ROLE RESTAURANT)
LEFT JOIN users ru
    ON ru.restaurant_id = r.id
   AND ru.role = 'restaurant'

WHERE o.order_id = ?;

  `,
      [orderID]
    );
  },
  async getByOrderByPhone(phone) {
    const [rows] = await db.query(
      `
    SELECT COUNT(*) AS count
    FROM orders o 
    WHERE o.customer_phone = ?
      AND o.order_status NOT IN ('completed', 'cancelled');
    `,
      [phone]
    );
    return rows[0]; // supaya langsung return { count: X }
  },

  // ❌ Batalkan order jika tidak ada driver yang menerima dalam 1 menit
  async cancelOrder(orderID, orderType) {
    let order_type;
    if (orderType == "ride") {
      order_status = "cancelled";
    } else {
      order_status = "process";
    }
    const sql = `
        UPDATE orders 
        SET order_status = '${order_status}', 
            cancelled_at = NOW(), 
            cancelled_reason = 'No driver accepted within 1 minute'
        WHERE order_id = ?
    `;
    await db.query(sql, [orderID]);
    console.log(
      `❌ Order ${orderID} dibatalkan oleh sistem (tidak ada driver).`
    );
    return true;
  },

  async verifyPIN(orderID, pin) {
    const [rows] = await db.query(
      "SELECT pin_code FROM orders WHERE order_id = ? AND pin_code = ?",
      [orderID, pin]
    );

    return rows.length > 0; // TRUE jika PIN cocok
  },
};
