// app/Helpers/FCMHelper.js
const admin = require("firebase-admin");

/**
 * Kirim notifikasi ke 1 device
 * @param {string} token - Token FCM penerima
 * @param {object} notification - Object { title, body }
 * @param {object} [data={}] - Data tambahan opsional
 * @returns {Promise<object>}
 */
async function sendFCM(token, notification = {}, data = {}) {
  const { title, body } = notification;

  // ✅ Validasi
  if (!token || !title || !body) {
    throw new Error("token, title, dan body wajib diisi!");
  }

  // ✅ Struktur pesan FCM
  const message = {
    token,
    notification: { title, body },
    android: {
      notification: {
        channelId: "alfago_channel", // harus sama dengan channel Android
        sound: "default",
        priority: "high",
        clickAction: data.click_action || "OPEN_ORDER_DETAIL",
      },
    },
    data: data || {},
  };

  try {
    const response = await admin.messaging().send(message);
    console.log("✅ Notifikasi terkirim ke FCM:", response);
    return { success: true, response };
  } catch (error) {
    console.error("❌ Gagal kirim FCM:", error);
    return { success: false, error: error.message };
  }
}

/**
 * Kirim notifikasi broadcast ke banyak device
 * @param {string[]} tokens - Array token penerima
 * @param {object} notification - Object { title, body }
 * @param {object} [data={}] - Data tambahan opsional
 * @returns {Promise<object>}
 */
async function sendFCMBroadcast(tokens = [], notification = {}, data = {}) {
  const { title, body } = notification;

  // ✅ Validasi
  if (!Array.isArray(tokens) || tokens.length === 0) {
    throw new Error("Tokens array tidak boleh kosong!");
  }
  if (!title || !body) {
    throw new Error("title dan body wajib diisi!");
  }

  // ✅ Struktur pesan multicast
  const message = {
    tokens,
    notification: { title, body },
    android: {
      notification: {
        channelId: "alfago_channel",
        sound: "default",
        priority: "high",
        clickAction: data.click_action || "OPEN_ORDER_DETAIL",
      },
    },
    data: data || {},
  };

  try {
    const response = await admin.messaging().sendEachForMulticast(message);
    console.log(
      `📣 Broadcast ke ${tokens.length} device. ✅ Sukses: ${response.successCount}, ❌ Gagal: ${response.failureCount}`
    );
    return { success: true, response };
  } catch (error) {
    console.error("❌ Gagal broadcast FCM:", error);
    return { success: false, error: error.message };
  }
}

module.exports = { sendFCM, sendFCMBroadcast };
