const User = require("../Models/UserModel");
const Log = require("../Models/LogModel");
const Order = require("../Models/OrderModel");
const RiwayatDriver = require("../Models/RiwayatDriverModel");
const { CodeClearCache } = require("../Models/ClearCacheModel");
const orderEvents = require("../Events/orderEvents");
const { sendFCM } = require("../Helpers/fcmHelper");

const connections = new Map(); // uuid => ws
let rejectedDrivers = new Set(); // gunakan Set biar cepat

function heartbeat() {
  this.isAlive = true;
}

// =======================
// IP BLOCKING & WAKTU
// =======================
const failedAttempts = new Map(); // ip => count
const blockedIPs = new Map(); // ip => expireTime
const IPLocalhost = "103.250.10.30";

function formatTanggalIndonesia(date = new Date()) {
  return (
    new Intl.DateTimeFormat("id-ID", {
      timeZone: "Asia/Jakarta",
      day: "2-digit",
      month: "2-digit",
      year: "numeric",
      hour: "2-digit",
      minute: "2-digit",
      second: "2-digit",
      hour12: false,
    })
      .format(date)
      .replace(/\./g, ":") + " WIB"
  );
}

function addFailedAttempt(ip) {
  const count = (failedAttempts.get(ip) || 0) + 1;
  failedAttempts.set(ip, count);

  if (count >= 5) {
    const blockUntil = Date.now() + 10 * 60 * 1000; // blokir 10 menit
    blockedIPs.set(ip, blockUntil);
    failedAttempts.delete(ip);
    console.log(
      `[${formatTanggalIndonesia()}] 🚫 IP ${ip} diblokir selama 10 menit karena 5 kali gagal login`
    );
  }
}

function isBlocked(ip) {
  const expire = blockedIPs.get(ip);
  if (!expire) return false;
  if (expire > Date.now()) return true;
  blockedIPs.delete(ip);
  return false;
}

// =======================
// HELPER
// =======================
function safeJsonParse(text) {
  if (typeof text !== "string") text = text.toString();
  try {
    return JSON.parse(text);
  } catch (e) {
    return null;
  }
}

// =======================
// MAIN EXPORT
// =======================
module.exports = {
  onConnection,
  startHeartbeat,
  sendToAll,
  sendToUuid,
  cleraCache,
  pushMessage,
};

// =======================
// ON CONNECTION
// =======================
function onConnection(ws) {
  let uuid = null;
  let phone = null;
  let token = null;
  let authKey = null;
  let userName = null;
  let action = null;
  let clientIP = null;

  ws.isAlive = true;
  ws.on("pong", heartbeat);

  // -----------------------
  // CEK KREDENSIAL
  // -----------------------
  async function cekKredensial(ws, authKey, phone, token) {
    try {
      clientIP = (ws?._socket?.remoteAddress || "unknown").replace(
        "::ffff:",
        ""
      );

      // console.log(`IP CLIENT: ${clientIP}`);

      if (
        !authKey ||
        typeof authKey !== "string" ||
        authKey.trim() !== process.env.API_SECRET_KEY
      ) {
        // if (isBlocked(clientIP)) {
        //   console.log(
        //     `[${formatTanggalIndonesia()}] 🚫 IP ${clientIP} diblokir, koneksi langsung ditolak.`
        //   );
        //   ws.send(
        //     JSON.stringify({
        //       type: "error",
        //       message: "Access denied: IP temporarily blocked",
        //     })
        //   );
        //   ws.close();
        //   return false;
        // }

        // Cek API KEY
        console.log(`❌ WebSocket Auth gagal: API Key tidak valid`);
        ws.send(
          JSON.stringify({
            type: "error",
            message: "Unauthorized: Invalid authKey",
          })
        );
        ws.close();
        addFailedAttempt(clientIP);
        console.log(
          `[${formatTanggalIndonesia()}] ❌ Unauthorized attempt from ${clientIP}, phone: ${
            phone || "-"
          }`
        );
        return false;
      }

      // Cek phone
      if (
        !phone ||
        typeof phone !== "string" ||
        phone.trim() === "" ||
        phone === "null" ||
        phone === "undefined"
      ) {
        console.log(`❌ WebSocket Auth gagal: phone kosong`);
        ws.send(
          JSON.stringify({
            type: "error",
            message: "Unauthorized: Phone required",
          })
        );
        ws.close();
        addFailedAttempt(clientIP);
        console.log(
          `[${formatTanggalIndonesia()}] ❌ Unauthorized attempt from ${clientIP}, phone: ${
            phone || "-"
          }`
        );
        return false;
      }

      // Cek token
      if (
        !token ||
        typeof token !== "string" ||
        token.trim() === "" ||
        token === "null" ||
        token === "undefined"
      ) {
        console.log(`❌ WebSocket Auth gagal: token kosong`);
        ws.send(
          JSON.stringify({
            type: "error",
            message: "Unauthorized: Token required",
          })
        );
        ws.close();
        addFailedAttempt(clientIP);
        console.log(
          `[${formatTanggalIndonesia()}] ❌ Unauthorized attempt from ${clientIP}, phone: ${phone}, token: ${token}`
        );
        return false;
      }

      // Cek phone & token di DB
      const [userRows] = await User.getByPhone(phone.trim());
      if (!userRows || userRows.length === 0) {
        console.log(`❌ WebSocket Auth gagal: Phone ${phone} tidak terdaftar`);
        ws.send(
          JSON.stringify({
            type: "error",
            message: "Unauthorized: User not found",
          })
        );
        ws.close();
        addFailedAttempt(clientIP);
        console.log(
          `[${formatTanggalIndonesia()}] ❌ Unauthorized attempt from ${clientIP}, phone: ${phone}`
        );
        return false;
      }

      const user = userRows[0];
      if (user.token !== token.trim()) {
        console.log(
          `❌ WebSocket Auth gagal: token tidak cocok untuk ${phone}`
        );
        ws.send(
          JSON.stringify({
            type: "error",
            message: "Unauthorized: Invalid token",
          })
        );
        ws.close();
        addFailedAttempt(clientIP);
        console.log(
          `[${formatTanggalIndonesia()}] ❌ Token mismatch attempt from ${clientIP}, phone: ${phone}`
        );
        return false;
      }

      return true;
    } catch (err) {
      console.error(
        `[${formatTanggalIndonesia()}] ⚠️ Error di cekKredensial():`,
        err
      );
      try {
        ws?.send(
          JSON.stringify({
            type: "error",
            message: "Internal server error during authentication",
          })
        );
        ws?.close();
      } catch {}
      return false;
    }
  }

  // ========================
  // WS MESSAGE HANDLER
  // ========================
  ws.on("message", async (rawMsg) => {
    try {
      const data = safeJsonParse(rawMsg);

      if (!data) return;

      authKey = data.authKey;
      uuid = data.uuid;
      phone = data.phone;
      token = data.token;
      action = data.action;
      const type = data.type;
      const msg = data.msg;

      const valid = await cekKredensial(ws, authKey, phone, token);
      if (!valid) {
        console.log(
          `🚫 Autentikasi gagal untuk ${
            phone || "unknown"
          } (${type}), proses dihentikan.`
        );
        return;
      }

      // ========= PING-PONG =========
      if (type === "ping") {
        ws.send(JSON.stringify({ type: "pong" }));
        return;
      }

      // ========= CLOSE =========
      if (type === "close") {
        if (uuid) connections.delete(uuid);

        if (!phone || phone === "null" || phone.trim() === "") {
          console.log("Guest close manually");
          return;
        }

        if (phone && token) {
          if (clientIP != IPLocalhost) {
            await User.setOffline(phone, token);
          }

          console.log(`❌ User closed manually: ${phone}`);
        }
        return;
      }

      // ========= OPEN =========
      if (type === "open") {
        const validOpen = await cekKredensial(ws, authKey, phone, token);
        if (!validOpen) {
          console.log(
            `❌ Autentikasi gagal saat OPEN dari ${phone || "unknown"}`
          );
          return;
        }

        const [userRows] = await User.getByPhone(phone);
        if (!userRows || userRows.length === 0) {
          console.log(`❌ Gagal OPEN: user ${phone} tidak ditemukan.`);
          ws.send(
            JSON.stringify({
              type: "error",
              message: "Unauthorized: User not found",
            })
          );
          ws.close();
          return;
        }

        const user = userRows[0];
        userName = user.nama;

        if (clientIP != IPLocalhost) {
          await User.setOnline(phone, token, uuid);
        }

        connections.set(uuid, ws);

        console.log(`✅ User online: ${phone}`);
        return;
      }

      // ========= UPDATE LOCATION =========
      if (type === "update_location") {
        const { lat, lng } = data;
        await User.updateLocation(phone, lat, lng);

        const [userRows] = await User.getByPhone(phone);
        if (!userRows.length) return;

        const role = userRows[0].role;
        const orderID = userRows[0].order_id;

        if (role === "driver" && orderID) {
          const [rows] = await Order.getByOrderID(orderID);
          if (rows.length > 0) {
            const { uuid_customer } = rows[0];
            module.exports.sendToUuid(uuid_customer, {
              type: "server_message",
              action: "driver_location",
              message: {
                orderID,
                driver_lat: lat,
                driver_lng: lng,
              },
            });
          }
        }
        return;
      }

      // ========= LOG =========
      if (type === "log") {
        const msgText = data.msg;
        await Log.insertLog(phone, msgText);
        return;
      }

      // ========= UPDATE rideToggle / rideFood =========
      if (type === "update") {
        const actionUpdate = data.action;
        const msgText = data.msg;

        if (actionUpdate === "rideToggle")
          await User.rideToggle(phone, msgText);
        if (actionUpdate === "rideFood") await User.rideFood(phone, msgText);

        console.log(`🔄 Update action: ${actionUpdate}`);
        return;
      }

      // ========= ORDER HANDLER (RIDE / FOOD) =========
      if (type === "order") {
        await handleOrderMessage({
          ws,
          data,
          msg,
          userName,
        });
        return;
      }

      // ========= CEK DATA =========
      if (type === "cek_data") {
        const [userRows] = await User.getByPhone(phone);
        if (userRows && userRows.length > 0) {
          const user = userRows[0];
          let data_order;
          if (user.order_id !== "") {
            const [row] = await Order.getByOrderID(user.order_id);
            data_order = row[0];
          }

          const role = userRows[0].role;
          const result = await Order.getByOrderByPhone(phone);
          const order_count = result.count;

          let messagePayload = { user, data_order, order_count };

          if (role === "driver") {
            const riwayat_driver = await RiwayatDriver.getRiwayat(phone);
            messagePayload.riwayat_driver = riwayat_driver;
          }

          module.exports.sendToUuid(user.uuid, {
            type: "server_message",
            action: type,
            message: messagePayload,
          });
        } else {
          console.log("❌ Tidak ada user ditemukan saat cek_data.");
        }
        return;
      }

      // ========= CACHE CLEAR =========
      if (type === "cache") {
        const code = await CodeClearCache();
        const [userRows] = await User.getByPhone(phone);
        if (userRows && userRows.length > 0) {
          const user = userRows[0];
          module.exports.sendToUuid(user.uuid, {
            type: "clear_cache",
            action: "",
            message: { code: code, msg: "Clear cache" },
          });
        } else {
          console.log("❌ Tidak ada user ditemukan saat cek_data.");
        }
        return;
      }

      console.log("❌ Tidak ada user ditemukan saat cek_data.");
    } catch (err) {
      console.error("WS Error", err);
    }
  });

  // ========================
  // CLOSE & ERROR HANDLER
  // ========================
  ws.on("close", async () => {
    if (uuid && connections.has(uuid)) {
      connections.delete(uuid);
    }

    if (!phone || phone === "null" || phone.trim() === "") {
      console.log("Guest disconnected");
      return;
    }

    if (phone && token) {
      if (clientIP != IPLocalhost) {
        await User.setOffline(phone, token);
      }
      console.log(`⚠️ User disconnected: ${phone}`);
    }
  });

  ws.on("error", (err) => {
    console.error(`WebSocket error: ${err.message}`);
  });
}

// ========================
// ORDER HANDLER (ASLINYA DI DALAM on("message"))
// ========================
async function handleOrderMessage({ ws, data, msg, userName }) {
  let payload = safeJsonParse(msg);
  if (!payload) {
    console.error("Pesan order tidak valid:", msg);
    return;
  }

  console.log("payload:", payload);

  const _dataAction = payload.action; // new / reject / accept / with_customer / completed
  const _action = data.action; // ride / food (jenis order)
  console.log("_action: ", _action);
  const customer_name = userName;
  const pickup = payload.lokasi_awal;
  const destination = payload.lokasi_akhir;
  const price = payload.biaya_total;
  const distance = payload.jarak;
  const duration = payload.waktu;
  const pickup_lat = payload.pickup_lat;
  const pickup_lng = payload.pickup_lng;
  const dst_lat = payload.dst_lat;
  const dst_lng = payload.dst_lng;
  const payment_metode = payload.payment_metode || payload.payment_method;
  const customer_phone = payload.customer_phone || payload.customerPhone;

  let driverPhone = null;
  let driverName = null;
  let driverTipeKendaraan = null;
  let driverNoPol = null;
  let driverLat = null;
  let driverLng = null;
  let customerLat = null;
  let customerLng = null;
  let UUIDDriver = null;
  let UUIDCustomer = null;
  let driverRating = null;
  let driverTokenFCM = null;
  let customerTokenFCM = null;
  let orderID = null;
  let restoPhone = null;
  let UUIDResto = null;
  let FCMResto = null;

  if (_action === "ride") {
    orderID = await Order.generateOrderId(_action);
  } else if (_action === "food") {
    orderID = payload.order_id || payload.orderID;

    const [resto] = await Order.getByOrderIDResto(orderID);
    console.log(`orderID: ${orderID}, resto[0]: ${resto[0]}`);
    FCMResto = resto[0].restaurant_fcm_token;
    UUIDResto = resto[0].restaurant_uuid;
    restoPhone = resto[0].restaurant_phone;

    // console.log(`orderResto: ${resto[0]}`);
    // console.log(`restoPhone: ${resto[0].loc_lat}, UUIDResto: ${UUIDResto}`);
  }

  // console.log(
  //   `RestaurantLocLat: ${resto[0].loc_lat}, RestaurantLocLng: ${RestaurantLocLng}`
  // );

  // status untuk loop
  let accepted = false;
  let startTime = Date.now();

  console.log(`CEK DATA CUSTOMER, ${customer_phone}`);
  const [userRowsCust] = await User.getByPhone(customer_phone);

  if (userRowsCust && userRowsCust.length > 0) {
    console.log("DATA CUSTOMER ADA");
    UUIDCustomer = userRowsCust[0].uuid;
    customerTokenFCM = userRowsCust[0].fcm_token;
  }
  console.log(`UUIDCustomer : ${UUIDCustomer}`);

  const [userRowsDriver] = await User.getByPhone(payload.driverPhone);
  if (userRowsDriver && userRowsDriver.length > 0) {
    driverName = userRowsDriver[0].nama;
  }

  // 🔁 LOOP CARI DRIVER
  async function findDriverLoop() {
    console.log(`▶️ findDriverLoop(${orderID}) dimulai`);

    let FCMSend;

    if (_action == "ride") {
      FCMSend = customerTokenFCM;
    } else if (_action == "food") {
      FCMSend = FCMResto;
    }

    console.log(`FCMSend: ${FCMSend}`);

    // timeout (30 detik)
    if (Date.now() - startTime > 60000) {
      console.log(
        "⏱️ Tidak ada driver yang terima dalam 60 detik. Order dibatalkan."
      );

      await Order.cancelOrder(orderID, _action);

      ws.send(
        JSON.stringify({
          type: "order_status",
          status: "cancelled",
          orderID,
        })
      );

      await sendFCM(
        FCMSend,
        {
          title: `Pesanan ${_action.toUpperCase()} anda dibatalkan`,
          body: `Driver tidak ada di daerah anda`,
        },
        {
          order_id: orderID,
          type: "new_order",
          click_action: "OPEN_ORDER_DETAIL",
        }
      );

      if (_action === "ride") {
        module.exports.sendToUuid(UUIDCustomer, {
          type: "server_message",
          action: "ride_order_cancel",
          message: {},
        });
      } else if (_action === "food") {
        module.exports.sendToUuid(UUIDResto, {
          type: "server_message",
          action: "food_order_cancel",
          message: {},
        });
      }

      return;
    }

    console.log(`payment_metode: ${payment_metode}`);
  let nearby;

  if (_action === "food") {
    nearby = await Order.nearbyDriver(
      customer_phone,
      _action,
      5000,
      payment_metode,
      payload.restaurant.loc_lat,
      payload.restaurant.loc_lng
    );
  } else if (_action === "ride") {
    nearby = await Order.nearbyDriver(
      customer_phone,
      _action,
      5000,
      payment_metode
    );
  }


    console.log(`nearby: ${nearby}`);

    if (!nearby || nearby.length === 0) {
      console.log("❌ Tidak ada driver dalam radius 5 km. Mencoba lagi...");
      // setTimeout(findDriverLoop, 5000);
      await Order.cancelOrder(orderID, _action);

      await sendFCM(
        FCMSend,
        {
          title: `Pesanan ${_action.toUpperCase()} anda dibatalkan`,
          body: `Driver tidak ada di daerah anda`,
        },
        {
          order_id: orderID,
          type: "new_order",
          click_action: "OPEN_ORDER_DETAIL",
        }
      );

      if (_action === "ride") {
        module.exports.sendToUuid(UUIDCustomer, {
          type: "server_message",
          action: "ride_order_cancel",
          message: {},
        });
      } else if (_action === "food") {
        module.exports.sendToUuid(UUIDResto, {
          type: "server_message",
          action: "food_order_cancel",
          message: {},
        });
      }
      return;
    }

    const availableDrivers = nearby.filter(
      (d) => !rejectedDrivers.has(d.phone)
    );

    if (availableDrivers.length === 0) {
      console.log(
        "🔁 Semua driver sudah pernah menolak. Reset daftar penolakan..."
      );
      rejectedDrivers.clear();
    }

    for (const driver of nearby) {
      if (accepted) break;
      if (rejectedDrivers.has(driver.phone)) {
        console.log(
          `⏭️ Skip driver ${driver.phone} karena sudah menolak sebelumnya.`
        );
        continue;
      }

      driverPhone = driver.phone;
      UUIDDriver = driver.uuid;
      driverName = driver.nama;
      driverTokenFCM = driver.fcm_token;

      if (_action === "ride") {
        await Order.accepttOrder(driverPhone, driverName, orderID, "new");
      } else if (_action === "food") {
        await Order.accepttOrder(driverPhone, driverName, orderID, "searching");
      }
      console.log(`🚀 Mengirim order ke driver ${driver.phone}`);
      console.log(`tokenFCMDriver: ${driverTokenFCM}`);
      // console.log("order_type", _order_type);

      const payload_ride = {
        orderID,
        customer_name,
        customer_phone,
        driverPhone,
        pickup,
        destination,
        price,
        distance,
        duration,
        pickup_lat,
        pickup_lng,
        dst_lat,
        dst_lng,
        payment_metode,
      };

      const payload_food = payload; // pastikan payload object, bukan string

      // console.log("_order_type:", _order_type);

      let data_payload, action_payload;

      if (_action === "ride") {
        data_payload = payload_ride;
        action_payload = "ride_order_new";
      } else if (_action === "food") {
        data_payload = payload_food;
        action_payload = "food_order_new";
      } else {
        console.log("⚠️ Unknown order type:", orderType);
        // return;
      }

      // if ((order_type = "ride")) {
      //   module.exports.sendToUuid(UUIDDriver, {
      //     type: "server_message",
      //     action: "ride_order_new",
      //     message: {
      //       orderID,
      //       customer_name,
      //       customer_phone,
      //       driverPhone,
      //       pickup,
      //       destination,
      //       price,
      //       distance,
      //       duration,
      //       pickup_lat,
      //       pickup_lng,
      //       dst_lat,
      //       dst_lng,
      //       payment_metode,
      //     },
      //   });
      // } else if ((order_type = "food")) {
      // }
      // console.log("action_payload:", action_payload);
      module.exports.sendToUuid(UUIDDriver, {
        type: "server_message",
        action: _action === "food" ? "food_order_new" : "ride_order_new",
        message: data_payload,
      });

      await sendFCM(
        driverTokenFCM,
        {
          title: `Alfa ${_action}`,
          body: `Pesanan ${_action} baru #${orderID}`,
        },
        {
          order_id: orderID,
          type: "new_order",
          click_action: "OPEN_ORDER_DETAIL",
        }
      );

      await new Promise((resolve) => {
        let responded = false;
        
        orderEvents.removeAllListeners(`reject_${orderID}`);
        orderEvents.removeAllListeners(`accept_${orderID}_${driver.phone}`);

        orderEvents.once(`reject_${orderID}`, async () => {
          console.log(`❌ Driver ${driver.phone} menolak order ${orderID}.`);
          await Order.accepttOrder(driverPhone, driverName, "", "0");
          rejectedDrivers.add(driver.phone);
          responded = true;
          resolve();
        });

        orderEvents.once(`accept_${orderID}_${driver.phone}`, async () => {
          console.log(`✅ Driver ${driver.phone} menerima order ${orderID}!`);
          accepted = true;
          responded = true;

          let st;

          if (_action === "food") {
            st = "driver";
          } else {
            st = "driver_to_customer";
          }
          await Order.accepttOrder(driver.phone, driverName, orderID, st);

          let fcm, t, uuid, ac, ms;
          if (_action === "food") {
            const [resto] = await Order.getByOrderIDResto(orderID);
            FCMResto = resto[0].restaurant_fcm_token;
            uuid = resto[0].restaurant_uuid;
          } else {
          }

          const [userRowsCust2] = await User.getByPhone(customer_phone);
          const [userRowsDriver2] = await User.getByPhone(driver.phone);

          driverName = userRowsDriver2[0].nama;
          driverTipeKendaraan = userRowsDriver2[0].driver_tipe_kendaraan;
          driverNoPol = userRowsDriver2[0].driver_no_pol;
          driverLat = userRowsDriver2[0].loc_lat;
          driverLng = userRowsDriver2[0].loc_lng;
          customerLat = userRowsCust2[0].loc_lat;
          customerLng = userRowsCust2[0].loc_lng;
          driverRating = userRowsDriver2[0].avg_rating_driver;
          driverTokenFCM = userRowsDriver2[0].fcm_token;

          console.log("customerTokenFCM:", FCMResto);

          if (_action === "food") {
            fcm = FCMResto;
            t = "Alfa Food";
            ac = "food_accept";
            ms = {};
          } else {
            fcm = customerTokenFCM;
            t = "Alfa Ride";
            uuid = UUIDCustomer;
            ac = "ride_accept";
            ms = {
              orderID,
              driverName,
              driverTipeKendaraan,
              driverNoPol,
              driverLat,
              driverLng,
              customerLat,
              customerLng,
              dst_lat,
              dst_lng,
              pickup,
              destination,
              price,
              driverRating,
            };
          }
          await sendFCM(
            fcm,
            {
              title: t,
              body: `Driver dengan nomor polisi ${driverNoPol} menuju ke lokasi kamu`,
            },
            {
              order_id: orderID,
              type: "new_order",
              click_action: "OPEN_ORDER_DETAIL",
            }
          );
          console.log("UUID:", uuid);
          module.exports.sendToUuid(uuid, {
            type: "server_message",
            action: ac,
            message: ms,
          });

          resolve();
        });

        setTimeout(async () => {
          if (!responded) {
            await Order.accepttOrder(driverPhone, driverName, "", "0");
            console.log(
              `⌛ Driver ${driver.phone} tidak merespons order ${orderID}.`
            );
            rejectedDrivers.add(driver.phone);
            resolve();
          }
        }, 15000);
      });
    }

    if (!accepted) {
      console.log("🔄 Tidak ada driver yang menerima, mencoba lagi...");
      setTimeout(findDriverLoop, 3000);
    }
  }

  console.log("_dataAction:", _dataAction);
  // Eksekusi sesuai action
  if (_dataAction === "new") {
    await Order.insertOrder(
      orderID,
      _action,
      customer_phone,
      customer_name,
      null,
      pickup,
      destination,
      pickup_lat,
      pickup_lng,
      dst_lat,
      dst_lng,
      distance,
      duration,
      price,
      "",
      payment_metode
    );

    findDriverLoop();
  } else if (_dataAction === "food") {
    // console.log("dataAction === food");
    console.log("PESANAN SELESAI, MENCARI DRIVER");
    module.exports.sendToUuid(UUIDCustomer, {
      type: "server_message",
      action: "resto_ready_order",
      message: { msg: "Mencari driver" },
    });
    findDriverLoop();
  } else if (_dataAction === "resto_accept_order") {
    console.log("PESANAN DI TERIMA RESTO");
    module.exports.sendToUuid(UUIDCustomer, {
      type: "server_message",
      action: "resto_accept_order",
      message: { msg: "Pesanan diterima restoran" },
    });
  } else if (_dataAction === "reject") {
    console.log(`TOLAK ORDERAN ${payload.orderID}`);
    await Order.rejectOrder(driverPhone, payload.orderID);
    orderEvents.emit(`reject_${payload.orderID}`);
  } else if (_dataAction === "accept") {
    console.log(payload);
    accepted = true;
    console.log("DRIVER MENUJU RESTORAN");
    module.exports.sendToUuid(UUIDCustomer, {
      type: "server_message",
      action: "driver_to_restaurant",
      message: { msg: "Driver menuju restoran" },
    });
    console.log(`accept_${payload.orderID}_${payload.driverPhone}`);
    orderEvents.emit(`accept_${payload.orderID}_${payload.driverPhone}`);
  } else if (_dataAction === "deliver") {
    // const isValid = await Order.verifyPIN(payload.orderID, payload.pin);
    // console.log("PIN check result:", isValid);

    const [resto] = await Order.getByOrderIDResto(payload.orderID);
    FCMResto = resto[0].restaurant_fcm_token;
    uuid = resto[0].restaurant_uuid;

    // setTimeout(() => {
    //   if (!isValid) {
    //     module.exports.sendToUuid(uuid, {
    //       type: "server_message",
    //       action: "pin_invalid",
    //       message: { error: "PIN salah!" },
    //     });
    //     return;
    //   }
    // }, 1500);

    // // PIN benar → update status
    await Order.accepttOrder("", "", payload.orderID, "deliver", "");
    console.log("UUIDDriver:", uuid);

    console.log(`SEDANG MENGANTAR PESANAN, ${UUIDCustomer}`);
    module.exports.sendToUuid(UUIDCustomer, {
      type: "server_message",
      action: "on_delivery",
      message: { msg: "Pesanan sedang di antar" },
    });

    module.exports.sendToUuid(uuid, {
      type: "server_message",
      action: "pin_valid",
      message: { msg: "PIN Valid" },
    });

    // setTimeout(() => {

    // }, 1500);
  } else {
    console.log(UUIDCustomer);
    console.log(`PRICE: ${payload.tarif}`);

    await Order.accepttOrder(
      payload.driverPhone,
      driverName,
      payload.orderID,
      _dataAction,
      price
    );

    if (_dataAction === "with_customer") {
      await sendFCM(
        customerTokenFCM,
        {
          title: "Alfa Ride",
          body: `Anda bersama dengan driver ${driverName}`,
        },
        {
          order_id: orderID,
          type: "new_order",
          click_action: "OPEN_ORDER_DETAIL",
        }
      );

      module.exports.sendToUuid(UUIDCustomer, {
        type: "server_message",
        action: _dataAction,
        message: {},
      });
    } else if (_dataAction === "completed") {
      if (_action === "food") {
        console.log("ORDER FOOD SELESAI");

        const [resto] = await Order.getByOrderIDResto(payload.orderID);
        FCMResto = resto[0].restaurant_fcm_token;
        uuid = resto[0].restaurant_uuid;

        // setTimeout(() => {
        //   if (!isValid) {
        //     module.exports.sendToUuid(uuid, {
        //       type: "server_message",
        //       action: "pin_invalid",
        //       message: { error: "PIN salah!" },
        //     });
        //     return;
        //   }
        // }, 1500);

        // // PIN benar → update status
        await Order.accepttOrder("", "", payload.orderID, "completed", "");
        console.log("UUIDDriver:", uuid);
        module.exports.sendToUuid(UUIDCustomer, {
          type: "server_message",
          action: "completed",
          message: { msg: "Order completed" },
        });
        module.exports.sendToUuid(uuid, {
          type: "server_message",
          action: "completed",
          message: { msg: "Order completed" },
        });

        await sendFCM(
          FCMResto,
          {
            title: "Alfa Food",
            body: "Pesanan anda sudah selesai.",
          },
          {
            order_id: orderID,
            type: "new_order",
            click_action: "OPEN_ORDER_DETAIL",
          }
        );
      } else {
        await sendFCM(
          customerTokenFCM,
          {
            title: "Alfa Ride",
            body: "Perjalanan anda sudah selesai.",
          },
          {
            order_id: orderID,
            type: "new_order",
            click_action: "OPEN_ORDER_DETAIL",
          }
        );

        module.exports.sendToUuid(UUIDCustomer, {
          type: "server_message",
          action: _dataAction,
          message: { order_id: payload.orderID },
        });
      }
    }
  }
}

console.log("CACHE COUNT:", Object.keys(require.cache).length);

// ========================l
// HEARTBEAT
// ========================
function startHeartbeat(wss, interval = 5000) {
  setInterval(() => {
    wss.clients.forEach((ws) => {
      if (ws.isAlive === false) {
        ws.terminate();
      } else {
        ws.isAlive = false;
        ws.ping();
      }
    });
  }, interval);
}

// ========================
// SEND UTILITIES
// ========================
function sendToAll(message) {
  connections.forEach((ws) => {
    if (ws.readyState === ws.OPEN) {
      ws.send(JSON.stringify(message));
      console.log(JSON.stringify(message));
    }
  });
}

function sendToUuid(uuid, message) {
  const ws = connections.get(uuid);
  if (ws && ws.readyState === ws.OPEN) {
    ws.send(JSON.stringify(message));
    console.log(`sendToUuid: ${JSON.stringify(message)}`);
    return true;
  }
  return false;
}

function cleraCache(message) {
  connections.forEach((ws) => {
    if (ws.readyState === ws.OPEN) {
      ws.send(JSON.stringify(message));
      console.log(JSON.stringify(message));
    }
  });
}

// ========================
// HTTP PUSH MESSAGE
// ========================
async function pushMessage(req, res) {
  const { uuid, action, message } = req.body;
  console.log(req.body);

  if (!message) {
    return res
      .status(400)
      .json({ success: false, message: "Message is required" });
  }

  if (uuid) {
    const sent = sendToUuid(uuid, {
      type: "server_message",
      action: action,
      message,
    });
    if (!sent)
      return res
        .status(404)
        .json({ success: false, message: "UUID not found or offline" });
    return res.json({
      success: true,
      message: "Message sent to specific user",
    });
  } else {
    sendToAll({ type: "server_broadcast", message });
    return res.json({
      success: true,
      message: "Broadcast sent to all users",
    });
  }
}
