document.addEventListener("DOMContentLoaded", async () => {
  AndroidApp.showBottomNav();
  const container = document.getElementById("restaurants-list");
  const searchBox = document.getElementById("searchBox");
  const searchBtn = document.querySelector(".search-btn");
  const locationText = document.getElementById("current-location");

  const tplCard = document.getElementById("template-restaurant-card");
  const tplNoData = document.getElementById("template-no-data");
  const tplCart = document.getElementById("template-cart-overlay");

  async function fetchRestaurants(keyword = "") {
    try {
      const userLat = window.Latitude || "";
      const userLon = window.Longitude || "";

      let url = keyword
        ? `/api/alfafood/restaurants?search=${encodeURIComponent(keyword)}`
        : `/api/alfafood/restaurants?token=${token}&lat=${userLat}&lng=${userLon}`;

      const res = await fetch(url);
      const json = await res.json();

      const restaurants = Array.isArray(json.data) ? json.data : [];

      if (restaurants.length === 0) {
        container.innerHTML = "";
        container.appendChild(tplNoData.content.cloneNode(true));
        return;
      }

      renderRestaurants(restaurants);
    } catch (err) {
      console.error("Gagal load restoran:", err);
    }
  }

  function renderRestaurants(restaurants) {
    container.innerHTML = "";

    restaurants.forEach((resto) => {
      const node = tplCard.content.cloneNode(true);

      const card = node.querySelector(".restaurant-card");
      const img = node.querySelector(".resto-image");
      const name = node.querySelector(".resto-name");
      const desc = node.querySelector(".resto-desc");
      const ratingBox = node.querySelector(".rating-under");
      const ratingLine = node.querySelector(".rating-line");
      const ratingValue = node.querySelector(".rating-value");
      const closedLabel = node.querySelector(".closed-label");

      const isClosed = resto.open === "0";

      img.src =
        resto.image_url ||
        "https://img.icons8.com/ios-filled/100/ffffff/restaurant.png";
      name.textContent = resto.name;
      desc.textContent = resto.description || "Deskripsi tidak tersedia";

      const distanceText = node.querySelector(".distance-text");

      if (resto.loc_lat && resto.loc_lng && Latitude && Longitude) {
        const dist = calculateDistance(
          parseFloat(Latitude),
          parseFloat(Longitude),
          parseFloat(resto.loc_lat),
          parseFloat(resto.loc_lng)
        );

        distanceText.textContent =
          dist < 1 ? `${Math.round(dist * 1000)} m` : `${dist.toFixed(1)} km`;
      } else {
        distanceText.textContent = "";
      }

      if (resto.rating && parseFloat(resto.rating) > 0) {
        ratingValue.textContent = parseFloat(resto.rating).toFixed(1);
        ratingBox.classList.remove("d-none");
      } else if (distanceText.textContent) {
        ratingBox.classList.remove("d-none"); // tampilkan jarak walau tanpa rating
      } else {
        ratingBox.classList.add("d-none");
      }

      if (resto.rating && parseFloat(resto.rating) == 0) {
        ratingLine.classList.add("hidden");
      }

      if (isClosed) {
        card.classList.add("closed");
        closedLabel.classList.remove("d-none");
      } else {
        card.addEventListener("click", () => {
          if (window.AndroidApp?.hideBottomNav) {
            window.AndroidApp.hideBottomNav();
          }
          window.location.href = `/alfafood/${resto.id}/menus`;
        });
      }

      container.appendChild(node);
    });
  }
  function waitForLocation(callback) {
    const interval = setInterval(() => {
      if (window.Latitude && window.Longitude) {
        clearInterval(interval);
        callback();
      }
    }, 200);
  }

  // SEARCH
  let searchTimeout = null;
  searchBox.addEventListener("input", () => {
    const keyword = searchBox.value.trim();

    clearTimeout(searchTimeout);

    searchBtn.innerHTML = keyword
      ? `<i class="fa-solid fa-xmark"></i>`
      : `<i class="fa-solid fa-magnifying-glass"></i>`;

    if (keyword.length === 0) {
      searchTimeout = setTimeout(() => fetchRestaurants(""), 300);
      return;
    }

    if (keyword.length < 3) {
      container.innerHTML = `<p style="text-align:center;margin-top:40px">✍️ Minimal 3 huruf...</p>`;
      return;
    }

    searchTimeout = setTimeout(() => fetchRestaurants(keyword), 300);
  });

  searchBtn.addEventListener("click", () => {
    if (searchBox.value.trim() !== "") {
      searchBox.value = "";
      searchBtn.innerHTML = `<i class="fa-solid fa-magnifying-glass"></i>`;
      fetchRestaurants("");
    }
  });

  // Active cart overlay
  function showActiveCartOverlay() {
    const savedCart = JSON.parse(localStorage.getItem("cartData") || "{}");
    if (!savedCart.items || savedCart.items.length === 0) return;

    if (document.getElementById("activeCartOverlay")) return;

    const totalItems = savedCart.items.reduce((s, i) => s + i.qty, 0);
    const subtotal = savedCart.items.reduce((s, i) => s + i.qty * i.price, 0);

    const node = tplCart.content.cloneNode(true);
    const overlay = node.querySelector("#activeCartOverlay");

    overlay.querySelector(".cart-resto-name").textContent =
      savedCart.restaurant_name;
    overlay.querySelector(
      ".cart-info-small"
    ).textContent = `${totalItems} item • Rp ${subtotal.toLocaleString()}`;

    overlay.addEventListener("click", () => {
      window.location.href = `/alfafood/${savedCart.restaurant_id}/menus`;
    });

    document.body.appendChild(overlay);

    setTimeout(() => overlay.classList.add("show"), 150);
  }

  await fetchRestaurants();
  showActiveCartOverlay();
  loadGoogleMaps();
  waitUntilMapReady(() => getNamaTempat());
  async function getNamaTempat() {
    const getLocation = await getCurrentLocationFetchOnly();
    const status = getLocation.status;

    if (status === "OK" && getLocation.results.length > 0) {
      const fullAddress = getLocation.results[0].formatted_address;
      // alert(fullAddress);

      const parts = fullAddress.split(",");

      // regex plus code (contoh: 39FQ+87M)
      const plusCodeRegex = /^[A-Z0-9]{4}\+[A-Z0-9]{2,}$/i;

      let shortAddress = parts[0].trim();

      // kalau alamat pertama adalah plus code → pakai alamat kedua
      if (plusCodeRegex.test(shortAddress) && parts.length > 1) {
        shortAddress = parts[1].trim();
      }

      // Ambil bagian pertama sebelum koma
      // const shortAddress = fullAddress.split(",")[0];
      if (locationText) locationText.textContent = shortAddress;
    } else {
      if (locationText) locationText.textContent = locationText.textContent;
    }
    try {
    } catch (err) {
      // console.error("Error geocoding:", err);
      NotifAndroidApp("Error geocoding", err);
      // if (locationText)
      //     locationText.textContent = `Lat: ${Latitude}, Lon: ${Longitude}`;
    }
  }

  function calculateDistance(lat1, lon1, lat2, lon2) {
    const R = 6371; // km
    const dLat = ((lat2 - lat1) * Math.PI) / 180;
    const dLon = ((lon2 - lon1) * Math.PI) / 180;

    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos((lat1 * Math.PI) / 180) *
        Math.cos((lat2 * Math.PI) / 180) *
        Math.sin(dLon / 2) *
        Math.sin(dLon / 2);

    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c; // km
  }
});
