<?php

namespace App\Models;

use CodeIgniter\Model;
use LDAP\Result;

class OrderModel extends Model
{
    protected $table = 'orders';
    protected $primaryKey = 'order_id';
    protected $allowedFields = [
        'order_id',
        'order_type',
        'restaurant_id',
        'pin_code',
        'customer_name',
        'customer_phone',
        'driver_phone',
        'pickup',
        'destination',
        'distance',
        'duration',
        'pickup_lat',
        'pickup_lng',
        'destination_lat',
        'destination_lng',
        'destination_address',
        'tarif',
        'order_status',   // pastikan nama kolom di DB memang 'status'
        'jenis',
        'created_at',
        'payment_metode',
        'updated_at'
    ];

    // Ambil semua order, terbaru di atas
    public function getAllOrders()
    {
        return $this->orderBy('created_at', 'DESC')->findAll();
    }

    // Ambil semua order berdasarkan nomor telepon (bisa customer atau driver)
    // token parameter aku hapus dari query karena tidak ada kolom token di table orders.
    // Jika memang mau autentikasi pakai token, lakukan validasi di controller sebelum memanggil model.



    // Ambil 1 order by order_id (simple)
    public function getOrderById($orderId)
    {
        return $this->where('order_id', $orderId)->first();
    }
    // Ambil order plus info driver (lokasi, tipe kendaraan, no_pol)
    public function getOrderWithDriver(string $orderId)
    {
        return $this->select('orders.*, 
                              users.driver_tipe_kendaraan,
                              users.driver_no_pol,
                              users.loc_lat AS driver_loc_lat,
                              users.loc_lng AS driver_loc_lng')
            // join users USING driver_phone -> users.phone = orders.driver_phone
            ->join('users', 'users.phone = orders.driver_phone', 'left')
            ->where('orders.order_id', $orderId)
            ->first();
    }
    // Customer History
    public function getAllOrdersByPhone($phone)
    {
        return $this->where('customer_phone', $phone)
            ->orderBy('created_at', 'DESC')
            ->findAll();
    }
    // Driver History
    public function getAllOrdersByDriverPhone($phone)
    {
        return $this->where('driver_phone', $phone)
            ->orderBy('created_at', 'DESC')
            ->findAll();
    }
    // Restaurant Orders
    public function getOrdersByRestaurant0($restaurantId)
    {
        return $this->where('id', $restaurantId)
            ->orderBy('created_at', 'DESC')
            ->findAll();
    }
    public function getOrdersByRestaurant1($restaurantId)
    {
        // JOIN orders + restaurants + order_items
        $builder = $this->db->table('orders o')
            ->select('
            o.order_id AS order_id,
            o.customer_name,
            o.driver_name,
            o.pin_code,
            o.payment_metode AS payment_method,
            o.tarif AS price,
            o.created_at AS datetime,
            o.status,
            
            r.name AS restaurant_name,

            oi.menu_name AS item_name,
            oi.qty,
            oi.note
        ')
            ->join('restaurants r', 'r.id = o.restaurant_id')     // ✔ CORRECT
            ->join('order_items oi', 'oi.order_id = o.id')         // ✔ CORRECT
            ->where('o.restaurant_id', $restaurantId)              // ✔ CORRECT
            ->orderBy('o.status', 'new', 'process', 'driver', 'deliver', 'completed', 'o.id DESC');

        $result = $builder->get()->getResultArray();

        // ==========================
        // Group items per order
        // ==========================
        $orders = [];

        foreach ($result as $row) {
            $id = $row['order_id'];

            // Jika order belum dibuat
            if (!isset($orders[$id])) {
                $orders[$id] = [
                    'id'             => $row['order_id'],
                    'customer_name'  => $row['customer_name'],
                    'driver_name'    => $row['driver_name'],
                    'pin_code'       => $row['pin_code'],
                    'payment_method' => $row['payment_method'],
                    'price'          => $row['price'],
                    'datetime'       => $row['datetime'],
                    'status'         => $row['status'],
                    'restaurant_name' => $row['restaurant_name'],

                    'items' => []
                ];
            }

            // Tambahkan item ke dalam array items
            $orders[$id]['items'][] = [
                'name' => $row['item_name'],
                'qty'  => $row['qty'],
                'note' => $row['note']
            ];
        }

        // Kembalikan array numerik
        return array_values($orders);
    }
    public function getOrdersByRestaurant2()
    {
        $orders = [
            [
                "id" => "INV202401",
                "customer_name" => "Budi Pratama",
                "driver_name" => null,
                "pin_code" => null,
                "payment_method" => "Tunai",
                "items" => [
                    ["name" => "Ayam Geprek", "qty" => 1, "note" => "Pedas level 3\nJangan terlalu asin\nSambal dipisah"],
                    ["name" => "Es Teh Manis", "qty" => 2, "note" => "Kurangi gula"]
                ],
                "price" => 28000,
                "datetime" => "2025-01-20 14:32",
                "status" => "waiting"
            ],

            [
                "id" => "INV202402",
                "customer_name" => "Siti Aminah",
                "driver_name" => null,
                "pin_code" => null,
                "payment_method" => "QRIS",
                "items" => [
                    ["name" => "Soto Lamongan", "qty" => 1, "note" => ""],
                    ["name" => "Kerupuk",       "qty" => 1, "note" => ""]
                ],
                "price" => 22000,
                "datetime" => "2025-01-20 15:12",
                "status" => "process"
            ],

            [
                "id" => "INV202403",
                "customer_name" => "Dedy Kurniawan",
                "driver_name" => "Driver Ahmad",
                "pin_code" => "4821",
                "payment_method" => "Tunai",
                "items" => [
                    ["name" => "Bakso Jumbo", "qty" => 2, "note" => "Tidak pakai daun bawang"]
                ],
                "price" => 40000,
                "datetime" => "2025-01-20 15:50",
                "status" => "driver"
            ],

            [
                "id" => "INV202405",
                "customer_name" => "Rahmat Hidayat",
                "driver_name" => "Driver Asep",
                "pin_code" => "3348",
                "payment_method" => "Tunai",
                "items" => [
                    ["name" => "Nasi Goreng Spesial", "qty" => 1, "note" => ""],
                    ["name" => "Teh Tarik",            "qty" => 1, "note" => "Sambal terpisah ya"]
                ],
                "price" => 30000,
                "datetime" => "2025-01-20 16:45",
                "status" => "done"
            ],
        ];

        return $orders;
    }
    public function getOrdersByRestaurant3($restaurantId)
    {
        $builder = $this->db->table('orders o')
            ->select('
            o.order_id AS order_id,
            o.customer_name,
            o.driver_name,
            o.pin_code,
            o.payment_metode AS payment_method,
            o.tarif AS price,
            o.created_at AS datetime,
            o.order_status,

            r.name AS restaurant_name,

            oi.menu_name AS item_name,
            oi.qty,
            oi.note
        ')
            ->join('restaurants r', 'r.id = o.restaurant_id')
            ->join('order_items oi', 'oi.order_id = o.id')
            ->where('o.restaurant_id', $restaurantId)
            ->where('o.order_type', 'food')

            // ⬇️ ORDER STATUS CUSTOM URUTAN WORKFLOW
            ->orderBy("FIELD(o.order_status, 'new', 'process', 'driver', 'deliver', 'completed')")

            // ⬇️ DI DALAM STATUS YANG SAMA, URUTKAN BERDASARKAN ID DESC
            ->orderBy('o.id', 'DESC');

        $result = $builder->get()->getResultArray();

        // ==========================
        // Group items per order
        // ==========================
        $orders = [];

       
        foreach ($result as $row) {
            $id = $row['order_id'];

            if (!isset($orders[$id])) {
                $orders[$id] = [
                    'id'             => $row['order_id'],
                    'customer_name'  => $row['customer_name'],
                    'driver_name'    => $row['driver_name'],
                    'pin_code'       => $row['pin_code'],
                    'payment_method' => $row['payment_method'],
                    'price'          => $row['price'],
                    'datetime'       => $row['datetime'],
                    'status'         => $row['order_status'],
                    'restaurant_name' => $row['restaurant_name'],
                    'items'          => []
                ];
            }

            $orders[$id]['items'][] = [
                'name' => $row['item_name'],
                'qty'  => $row['qty'],
                'note' => $row['note']
            ];
        }

        return array_values($orders);
    }
    public function getOrders($restaurantId = null, $orderId = null)
    {
        $builder = $this->db->table('orders o')
            ->select('
            o.order_type AS order_type,
            o.restaurant_id AS restaurant_id,
            o.order_id AS order_id,
            o.customer_phone,
            o.customer_name,
            o.driver_phone,
            o.driver_name,
            o.pin_code,
            o.payment_metode AS payment_method,
            o.tarif AS price,
            o.created_at AS datetime,
            o.order_status,
            o.created_at,
            o.pickup_lat,
            o.pickup_lng,
            o.destination_lat,
            o.destination_lng,
            o.destination_address,
            o.pickup_address,

            r.name AS restaurant_name,
            r.address AS restaurant_address,

            oi.menu_name AS item_name,
            oi.qty,
            oi.note
        ')
            ->join('restaurants r', 'r.id = o.restaurant_id')
            ->join('order_items oi', 'oi.order_id = o.id')
            ->where('o.order_type', 'food');

        // ==========================
        // FILTER DINAMIS
        // ==========================
        if (!empty($restaurantId)) {
            $builder->where('o.restaurant_id', $restaurantId);
        }

        if (!empty($orderId)) {
            $builder->where('o.order_id', $orderId);
        }

        // ==========================
        // ORDERING SAMA PERSIS
        // ==========================
        $builder->orderBy("
            FIELD(o.order_status, 
                'new', 
                'process', 
                'driver', 
                'deliver', 
                'completed',
                'cancelled'
            )
        ", '', false);

        $builder->orderBy('o.id', 'DESC');


        


        $result = $builder->get()->getResultArray();

        // ==========================
        // GROUPING SAMA PERSIS
        // ==========================
        $orders = [];

        foreach ($result as $row) {
            $id = $row['order_id'];

            if (!isset($orders[$id])) {
                $orders[$id] = [                    
                    'action'          => $row['order_type'],
                    'restaurant_id'   => $row['restaurant_id'],
                    'order_id'        => $row['order_id'],
                    'customer_phone'  => $row['customer_phone'],
                    'customer_name'   => $row['customer_name'],
                    'driver_phone'    => $row['driver_phone'],
                    'driver_name'     => $row['driver_name'],
                    'pin_code'        => $row['pin_code'],
                    'payment_method'  => $row['payment_method'],
                    'price'           => $row['price'],
                    'datetime'        => $row['datetime'],
                    'status'          => $row['order_status'],
                    'restaurant_name' => $row['restaurant_name'],
                    'created_at'      => $row['created_at'],
                    'pickup_lat'      => $row['pickup_lat'],
                    'pickup_lng'      => $row['pickup_lng'],
                    'destination_lat' => $row['destination_lat'],
                    'destination_lng' => $row['destination_lng'],
                    'destination_address'  => $row['destination_address'],
                    'restaurant_address'   => $row['restaurant_address'],
                    'pickup_address'   => $row['pickup_address'],
                    'items'           => []
                ];
            }

            $orders[$id]['items'][] = [
                'name' => $row['item_name'],
                'qty'  => $row['qty'],
                'note' => $row['note']
            ];
        }

        return array_values($orders);
    }
}
